'use strict';

var childProcess = require('child_process');
var path = require('path');
var shelljs = require('shelljs');

var util = require('./util.js');

var CWD = process.cwd();
var TEMP = path.join(CWD, 'node_modules', '.npm-install-version-temp');

function install(npmPackage) {
  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  var _options$destination = options.destination,
      destination = _options$destination === undefined ? util.sanitize(npmPackage) : _options$destination,
      _options$overwrite = options.overwrite,
      overwrite = _options$overwrite === undefined ? false : _options$overwrite,
      _options$quiet = options.quiet,
      quiet = _options$quiet === undefined ? false : _options$quiet;


  var log = quiet ? function () {} : function () {
    var _console;

    return (_console = console).log.apply(_console, arguments);
  };

  if (!npmPackage) util.error();
  var destinationPath = path.join(CWD, 'node_modules', destination);
  if (!overwrite && util.directoryExists(destinationPath)) {
    return log('Directory at ' + destinationPath + ' already exists, skipping');
  }

  var errored = false;
  try {
    // make temp install dir
    shelljs.rm('-rf', TEMP);
    shelljs.mkdir('-p', path.join(TEMP, 'node_modules'));

    // copy local .npmrc file if exists
    var npmrcFile = path.join(CWD, '.npmrc');
    if (shelljs.test('-f', npmrcFile)) {
      shelljs.cp(npmrcFile, TEMP);
    }

    // install package to temp dir
    var installOptions = {
      cwd: TEMP,
      stdio: [null, null, null]
    };
    var command = process.platform === 'win32' ? 'npm.cmd' : 'npm';
    childProcess.spawnSync(command, ['install', npmPackage], installOptions);

    // get real package name
    var packageName = util.getPackageName(npmPackage);

    // move deps inside package
    shelljs.mkdir('-p', path.join(TEMP, 'node_modules', packageName, 'node_modules'));
    shelljs.ls(path.join(TEMP, 'node_modules')).forEach(function (dep) {
      if (dep === packageName) return;
      var from = path.join(TEMP, 'node_modules', dep).toString();
      var to = path.join(TEMP, 'node_modules', packageName, 'node_modules', dep).toString();
      shelljs.mv(from, to);
    });

    // copy to node_modules/
    shelljs.rm('-rf', destinationPath);
    shelljs.mv(path.join(TEMP, 'node_modules', packageName), destinationPath);

    log('Installed ' + npmPackage + ' to ' + destinationPath);
  } catch (err) {
    errored = true;
    console.error('Error installing ' + npmPackage);
    console.error(err.toString());
  } finally {
    // clean up temp install dir
    shelljs.rm('-rf', TEMP);

    if (errored) process.exit(1);
  }
}

module.exports = install;
//# sourceMappingURL=install.js.map