"use strict";

var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function (obj) { return typeof obj; } : function (obj) { return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj; };

var converters = module.exports = {};

var generateHeader = function generateHeader(repeat) {
    return function (input, json2md) {
        return "#".repeat(repeat) + " " + json2md(input);
    };
};

var indent = function indent(content, spaces, ignoreFirst) {
    var lines = content;

    if (typeof content === "string") {
        lines = content.split("\n");
    }

    if (ignoreFirst) {
        if (lines.length <= 1) {
            return lines.join("\n");
        }
        return lines[0] + "\n" + indent(lines.slice(1), spaces, false);
    }

    return lines.map(function (c) {
        return " ".repeat(spaces) + c;
    }).join("\n");
};

var parseTextFormat = function parseTextFormat(text) {

    var formats = {
        strong: "**",
        italic: "*"
    };

    return text.replace(/<\/?strong\>/gi, formats.strong).replace(/<\/?bold\>/gi, formats.strong).replace(/<\/?em\>/gi, formats.italic).replace(/<\/?italic\>/gi, formats.italic);
};

// Headings
converters.h1 = generateHeader(1);
converters.h2 = generateHeader(2);
converters.h3 = generateHeader(3);
converters.h4 = generateHeader(4);
converters.h5 = generateHeader(5);
converters.h6 = generateHeader(6);

converters.blockquote = function (input, json2md) {
    return json2md(input, "> ");
};

converters.img = function (input, json2md) {
    if (Array.isArray(input)) {
        return json2md(input, "", "img");
    }
    if (typeof input === "string") {
        return converters.img({ source: input, title: "" });
    }
    input.title = input.title || "";
    return "![" + input.title + "](" + input.source + ")";
};

converters.ul = function (input, json2md) {
    var c = "";
    for (var i = 0; i < input.length; ++i) {
        var marker = "";

        var type = Object.keys(input[i])[0];
        if (type !== "ul" && type !== "ol") {
            marker += "\n - ";
        }

        c += marker + parseTextFormat(indent(json2md(input[i]), 4, true));
    }
    return c;
};

converters.ol = function (input, json2md) {
    var c = "";
    var jumpCount = 0;
    for (var i = 0; i < input.length; ++i) {
        var marker = "";
        var type = Object.keys(input[i])[0];
        if (type !== "ul" && type !== "ol") {
            marker = "\n " + (i + 1 - jumpCount) + ". ";
        } else {
            jumpCount++;
        }

        c += marker + parseTextFormat(indent(json2md(input[i]), 4, true));
    }
    return c;
};

converters.code = function (input, json2md) {
    var c = "```" + (input.language || "") + "\n";
    if (Array.isArray(input.content)) {
        c += input.content.join("\n");
    } else {
        c += input.content;
    }
    c += "\n```";
    return c;
};

converters.p = function (input, json2md) {
    return parseTextFormat(json2md(input, "\n"));
};

converters.table = function (input, json2md) {

    if ((typeof input === "undefined" ? "undefined" : _typeof(input)) !== "object" || !input.hasOwnProperty("headers") || !input.hasOwnProperty("rows")) {
        return "";
    }

    var header = input.headers.join(" | "),
        spaces = input.headers.map(function () {
        return "---";
    }).join(" | "),
        data = input.rows.map(function (r) {
        return Array.isArray(r) ? r.map(function (el) {
            return parseTextFormat(json2md(el));
        }).join(" | ") : input.headers.map(function (h) {
            return parseTextFormat(json2md(r[h]));
        }).join(" | ");
    }).join("\n");

    return [header, spaces, data].join("\n");
};