"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
var json_stable_stringify_1 = __importDefault(require("json-stable-stringify"));
var quote_1 = __importDefault(require("quote"));
var ramda_1 = require("ramda");
var __1 = require("..");
var ticks = quote_1.default({ quotes: '`' });
exports.checkMark = '✔';
exports.emptyMark = '';
var isCustomFormat = function (formats) { return function (name) { return name in formats; }; };
var knownSchemaNames = function (schemas) { return __1.schemaNames(schemas); };
var isSchemaName = function (schemas) { return function (s) {
    return knownSchemaNames(schemas).includes(__1.normalizeName(s));
}; };
exports.anchor = function (s) { return ramda_1.toLower(s.replace(/[\.@]/g, '')); };
exports.anchorForSchema = function (s) {
    var schemaName = ramda_1.toLower(__1.normalizeName(s.schema.title));
    var seeVersion = __1.semverToString(s.version);
    var nameAndVersion = schemaName + "@" + seeVersion;
    return exports.anchor(nameAndVersion);
};
exports.enumToMarkdown = function (enumeration) {
    if (!enumeration) {
        return exports.emptyMark;
    }
    return ticks(enumeration.map(JSON.stringify).join(', '));
};
exports.formatToMarkdown = function (schemas, formats) { return function (value) {
    if (!value.format) {
        if (value.see) {
            if (typeof value.see === 'string') {
                return schemas && isSchemaName(schemas)(value.see)
                    ? "[" + value.see + "](#" + ramda_1.toLower(__1.normalizeName(value.see)) + ")"
                    : ticks(value.see);
            }
            else {
                var seeSchema = value.see;
                var schemaName = "" + seeSchema.schema.title;
                var seeVersion = __1.semverToString(seeSchema.version);
                var nameAndVersion = schemaName + "@" + seeVersion;
                var seeAnchor = exports.anchorForSchema(seeSchema);
                return schemas && isSchemaName(schemas)(schemaName)
                    ? "[" + nameAndVersion + "](#" + seeAnchor + ")"
                    : ticks(nameAndVersion);
            }
        }
        else {
            return exports.emptyMark;
        }
    }
    if (formats && isCustomFormat(formats)(value.format)) {
        return "[" + value.format + "](#formats)";
    }
    return ticks(value.format);
}; };
exports.findUsedColumns = function (headers, rows) {
    var isUsed = function (header) { return ramda_1.find(function (r) { return r[header]; }, rows); };
    var usedHeaders = headers.filter(isUsed);
    return usedHeaders;
};
exports.documentProperties = function (properties, required, schemas, formats) {
    if (required === void 0) { required = []; }
    var requiredProperties = Array.isArray(required)
        ? required
        : Object.keys(properties);
    var isRequired = function (name) { return requiredProperties.indexOf(name) !== -1; };
    var typeText = function (type) { return (Array.isArray(type) ? type.join(' or ') : type); };
    var deprecatedMessage = function (value) {
        return value.deprecated ? "**deprecated** " + value.deprecated : exports.emptyMark;
    };
    return Object.keys(properties)
        .sort()
        .map(function (prop) {
        var value = properties[prop];
        return {
            name: ticks(prop),
            type: typeText(value.type),
            required: isRequired(prop) ? exports.checkMark : exports.emptyMark,
            format: exports.formatToMarkdown(schemas, formats)(value),
            enum: exports.enumToMarkdown(value.enum),
            description: value.description ? value.description : exports.emptyMark,
            deprecated: deprecatedMessage(value),
        };
    });
};
exports.documentSchema = function (schema, schemas, formats) {
    var properties = schema.properties;
    if (properties) {
        var rows = exports.documentProperties(properties, schema.required, schemas, formats);
        var headers = [
            'name',
            'type',
            'required',
            'format',
            'enum',
            'description',
            'deprecated',
        ];
        var usedHeaders = exports.findUsedColumns(headers, rows);
        var table = [
            {
                table: {
                    headers: usedHeaders,
                    rows: rows,
                },
            },
        ];
        if (schema.additionalProperties) {
            table.push({
                p: 'This schema allows additional properties.',
            });
        }
        return table;
    }
    else {
        return { p: 'Hmm, no properties found in this schema' };
    }
};
var schemaNameHeading = function (name, version) {
    return name + "@" + version;
};
exports.documentObjectSchema = function (schema, schemas, formats) {
    var schemaName = schema.schema.title;
    if (schemaName.includes(' ')) {
        throw new Error("Schema title contains spaces \"" + schemaName + "\"\n      This can cause problems generating anchors!");
    }
    var schemaVersion = __1.semverToString(schema.version);
    var start = [
        { h3: schemaNameHeading(__1.normalizeName(schemaName), schemaVersion) },
    ];
    if (schema.package) {
        start.push({
            p: "Defined in " + ticks(schema.package),
        });
    }
    if (schema.schema.description) {
        start.push({ p: schema.schema.description });
    }
    if (schema.schema.deprecated) {
        start.push({
            p: "**deprecated** " + schema.schema.deprecated,
        });
    }
    var propertiesTable = exports.documentSchema(schema.schema, schemas, formats);
    var exampleFragment = ramda_1.flatten([
        { p: 'Example:' },
        {
            code: {
                language: 'json',
                content: json_stable_stringify_1.default(schema.example, { space: '  ' }),
            },
        },
    ]);
    return ramda_1.flatten(start.concat(propertiesTable).concat(exampleFragment));
};
